#include <assert.h>
#include "../common/Compatible.h"
#include "../common/misc.h"
#include "../common/Common.h"
#include "../common/debug.h"
#include "BVH.h"

static int nodeCount = 0;

void BVH::fillNode(BVHNode *n, int leftIndex, int rightIndex)
{
	nodeCount++;
	int numPrimitives = rightIndex - leftIndex;
	
	//encompass all the primitives
	for(unsigned int i=leftIndex; i < rightIndex; i++)
	{
		int primIndex = splitter->getPrimitiveIndex(i);
		PrimitiveBase *p = primitives[primIndex];
		n->encompass(p->bbMin, p->bbMax);
	}
	
	//leaf node case
	if(numPrimitives <= 1)
	{
		assert(numPrimitives > 0);
		n->type = BVHNode::leaf;
		
		int primIndex = splitter->getPrimitiveIndex(leftIndex);
		PrimitiveBase *p = primitives[primIndex];
		
		n->setPrimitive(p);
		return;
	}
	
	SplitData split = splitter->splitPrimitives(*n, leftIndex, rightIndex);
	
	//make children nodes
	n->setLeft(new BVHNode());
	fillNode(n->getLeft(), leftIndex, split.index+1);
	n->setRight(new BVHNode());
	fillNode(n->getRight(), split.index+1, rightIndex);
}

void BVH::build(PrimitivePtrList &pList)
{
	timeval start_time;
	timeval finish_time;
	
	printd(ALERT, "Building BVH of %i primitives.\n", pList.size());
	gettimeofday(&start_time, NULL);
	
	for(unsigned int i=0; i < pList.size(); i++)
	{
		primitives.push_back(pList[i]);
		this->box.encompass(pList[i]->bbMin, pList[i]->bbMax);
	}
	
	splitter = new SplitterType(primitives);
	rootNode = new BVHNode();

	fillNode(rootNode, 0, primitives.size());
	
	gettimeofday(&finish_time, NULL);
	int buildTime = timeval_to_millisecond(&start_time, &finish_time);
	printd(ALERT, "Created %i BVH nodes in %ims.\n", nodeCount, buildTime);
}

bool BVH::intersectNode(BVHNode *n, const Ray &r, HitPoint &h)
{
	#define showTraversalStats true
	
	if(showTraversalStats)
		h.r += 1.0f;
	if( !n->signedIntersect(r, h) )
		return false;
	
	if(n->type == BVHNode::internal) {
		bool hitLeft=false, hitRight=false;
		hitLeft = intersectNode((BVHNode*)n->getLeft(), r, h);
		hitRight = intersectNode((BVHNode*)n->getRight(), r, h);
		return hitLeft || hitRight;
	}
	
	if(showTraversalStats)
		h.b += 1.0f;
	if( n->getPrimitive()->rayIntersect(r, h) )
	{
		h.pos = r.parameter(h.distance);
		h.primitivePtr = n->getPrimitive();
		return true;
	}
	
	return false;
}

bool BVH::rayIntersectAll(const Ray &r, HitPoint &h)
{
	if(showTraversalStats)
	{
		h.r = 0.0f;
		h.g = 0.0f;
		h.b = 0.0f;
		h.skipShading = true;
	}
	
	bool hit = intersectNode(rootNode, r, h);
	if(showTraversalStats)
	{
		h.r *= 1.0f;
		h.b *= 10.0f;
	}
	return hit;
}

bool BVH::rayIntersectSingle(const Ray &r, HitPoint &h)
{
	bool hitSomething;
	//	intersects_calced++;
	
	hitSomething = h.primitivePtr->rayIntersect(r, h);
	if(hitSomething > EPSILON && hitSomething < maxFloat)
	{
		//h.distance = hitSomething;
		return true;
	}
	
	return false;
}
























