#ifndef __SPLITSELECTION
#define __SPLITSELECTION

#include "../common/Vector3.h"
#include "../common/SimpleBoundingBox.h"
#include "../Primitives/PrimitiveBase.h"
#include <algorithm>

class Centroid
{
public:
	int primitiveIndex;
	vector3 center;
};

bool compareX(const Centroid &c1, const Centroid &c2);
bool compareY(const Centroid &c1, const Centroid &c2);
bool compareZ(const Centroid &c1, const Centroid &c2);

typedef bool (*centroidComparer)(const Centroid&, const Centroid&);
extern centroidComparer centroidCompare[3];


class SplitData
{
public:
	int index;
	int axis;
	slimFloat position;
	slimFloat cost;
};

class SplitSelection
{
public:
	Centroid* centroids;

	SplitSelection(PrimitivePtrList &primitives)
	{
		centroids = new Centroid[primitives.size()];
		for(int i=0; i<primitives.size(); i++)
		{
			centroids[i].primitiveIndex = i;
			centroids[i].center = primitives[i]->center();
		}
	}

	int getPrimitiveIndex(int pos)
	{
		int index = centroids[pos].primitiveIndex;
		return index;
	}

protected:
	~SplitSelection()
	{
		delete[] centroids;
	}
};

class ObjectMedian : public SplitSelection
{
public:
	ObjectMedian(PrimitivePtrList& p) : SplitSelection(p) {}
	
	SplitData splitPrimitives(SimpleBoundingBox &box, int left, int right)
	{
		SplitData split;
		
		//pick an axis to split on
		vector3 boxSize = box.dimension();
		split.axis = boxSize.maxComponent();
		vector3 center = box.center();
		split.position = center[split.axis];
		
		std::sort(&centroids[left], &centroids[right], centroidCompare[split.axis]);
		
		//get the primitive that is next to the split
		split.index = -1;
		for(unsigned i=left; i<right; i++)
		{
			float primitivePosition = centroids[i].center[split.axis];
			if(primitivePosition < split.position)
				split.index = i;
		}
		
		//split didn't work, fill each with half
		if(split.index < left || split.index+1 >= right)
			split.index = left-1 + (right - left) / 2;
		
		return split;
	}
};

class SpatialMedian : public SplitSelection
{
public:
	SpatialMedian(PrimitivePtrList& p) : SplitSelection(p) {}

	SplitData splitPrimitives(SimpleBoundingBox &box, int left, int right)
	{
		SplitData split;

		//pick an axis to split on
		vector3 boxSize = box.dimension();
		split.axis = boxSize.maxComponent();
		
		std::sort(&centroids[left], &centroids[right], centroidCompare[split.axis]);

		//split is middle of list
		split.index = left-1 + (right - left) / 2;

		return split;
	}
};


class SAH : public SplitSelection
{
//from "Automatic Creation of Object Hierarchies for Radiosity Clustering"
//Mueller, Schaefer, Fellner
public:
	SAH(PrimitivePtrList& p) : SplitSelection(p)
	{
		leftArea = new slimFloat[p.size()];
		rightArea = new slimFloat[p.size()];
		primitives = &p;
	}

	SplitData splitPrimitives(SimpleBoundingBox &box, int left, int right)
	{
		float parentArea = box.surfaceArea();
		float recipParentArea = 1 / parentArea;
		SplitData bestSplit;
		bestSplit.axis = -1;
		bestSplit.index = -1;
		bestSplit.cost = maxFloat;

		
		for(int axis=0; axis<vector3::dim; axis++)
		{
			SimpleBoundingBox leftBox, rightBox;
			std::sort(&centroids[left], &centroids[right], centroidCompare[axis]);
			
			//get the areas for all left and right combinations
			for(int i=left; i<right-1; i++)
			{
				PrimitiveBase *leftPrim = primitives[0][centroids[i].primitiveIndex];
				leftBox.encompass(leftPrim->bbMin, leftPrim->bbMax);
				leftArea[i] = leftBox.surfaceArea();
			}

			for(int i=right-1; i>left; i--)
			{
				int rightIndex = i-1;
				PrimitiveBase *rightPrim = primitives[0][centroids[i].primitiveIndex];
				rightBox.encompass(rightPrim->bbMin, rightPrim->bbMax);
				rightArea[rightIndex] = rightBox.surfaceArea();
			}

			//combine left and right costs, pick best one
			for(int i=left; i<right-1; i++)
			{
				int leftNum = i-left + 1;
				int rightNum = right-1-i;
				float leftCost = leftNum * leftArea[i];
				float rightCost = rightNum * rightArea[i];
				float currentCost = recipParentArea * (leftCost + rightCost);

				if(currentCost < bestSplit.cost)
				{
					bestSplit.axis = axis;
					bestSplit.index = i;
					bestSplit.cost = currentCost;
				}
			}
		}

		//split didn't work, fill each with half
		if(bestSplit.index < left || bestSplit.index+1 >= right)
		{
			bestSplit.index = left-1 + (right - left) / 2;
		}
		else
		{
			//put the centroids back in the correct order
			std::sort(&centroids[left], &centroids[right], centroidCompare[bestSplit.axis]);
		}

		return bestSplit;
	}

private:
	slimFloat *leftArea, *rightArea;
	PrimitivePtrList *primitives;
};


#endif

