#include "../common/Timer.h"
#include "../common/misc.h"
#include "../common/Common.h"
#include "../common/debug.h"
#include "StackBVH.h"
#include <assert.h>

static int nodeCount = 0;

void StackBVH::fillNode(StackBVHNode *n, int leftIndex, int rightIndex)
{
	nodeCount++;
	int numPrimitives = rightIndex - leftIndex;
	assert(numPrimitives > 0);
	
	//encompass all the primitives
	for(int i=leftIndex; i < rightIndex; i++)
	{
		int primIndex = splitter->getPrimitiveIndex(i);
		PrimitiveBase *p = primitives[primIndex];
		n->encompass(p->bbMin, p->bbMax);
	}
	
	//leaf node case
	if(numPrimitives <= 1)
	{
		assert(numPrimitives > 0);
		//n->setNodeType(BVHNode::leaf);
		
		int primIndex = splitter->getPrimitiveIndex(leftIndex);
		PrimitiveBase *p = primitives[primIndex];
		
		n->setPrimitive(p);
		return;
	}
	
	SplitData split = splitter->splitPrimitives(*n, leftIndex, rightIndex);
	n->setAxis(split.axis);
	
	// Do the left part
	StackBVHNode *leftNode = getNextNodeFromList();
	StackBVHNode *rightNode = getNextNodeFromList();
	n->setLeft(leftNode);
	fillNode(leftNode, leftIndex, split.index+1);
	
	// Do the right part
	//n->setRight(rightNode); set by left automagically
	fillNode(rightNode, split.index+1, rightIndex);
}


void StackBVH::build(PrimitivePtrList &pList)
{
	Timer timer;
	
	printd(ALERT, "Building StackBVH of %i primitives.\n", pList.size());
	timer.start();
	
	for(unsigned int i=0; i < pList.size(); i++)
	{
		primitives.push_back(pList[i]);
		this->box.encompass(pList[i]->bbMin, pList[i]->bbMax);
	}
	
	//splitter = new SpatialMedian(primitives);
	splitter = new SplitterType(primitives);
	nodes = new StackBVHNode [ primitives.size() * 2 ];

	int addr = (int)nodes;
	int notAlign16 = addr & 0x0f;
	if( notAlign16 )
		printf("**************************************\n* not aligned on 16 byte boundaries! *\n**************************************\n");

	printf("%x %x %x\n", nodes, &nodes[1], &nodes[2]);
	
	fillNode(&nodes[0], 0, primitives.size());
	
	timer.stop();
	int buildTime = timer.inMilliseconds();
	printd(ALERT, "Created %i BVH nodes in %ims.\n", nodeCount, buildTime);
}

bool StackBVH::intersectNodesStack(StackBVHNode *rootNode, const Ray &r, HitPoint &h)
{
	// similar to "Ray tracing deformable scenes using dynamic bounding volume hierarchies", Wald
	#define showTraversalStats false
	int nodeTest = 0;
	int primTest = 0;
	Timer timer;
	
	if(showTraversalStats)
		timer.start();
	
	bool atLeastOneHit = false;
	SimpleStack<StackBVHNode*> stack;
	stack.push(NULL);
	
	StackBVHNode *currentNode = rootNode;
	
	while(true)
	{
		//bool hitNode = currentNode->fyffeIntersection(r, h);
		bool hitNode = currentNode->signedIntersect(r, h);
		if(showTraversalStats)
			nodeTest++;
		
		if(hitNode)
		{
			if(showTraversalStats && !currentNode->isInternalNode())
				primTest++;
			
			if(currentNode->isInternalNode())
			{
				// as suggested in "RT-DEFORM: Interactive Ray Tracing of Dynamic Scenes using BVHs", Lauterbach
				// similar to DSA "Ray Tracing with Reduced-Precision Bounding Volume Hierarchies", Mahovsky
				StackBVHNode *left = currentNode->getLeft();
				int axis = currentNode->getAxis();
				int nextNodeOffset = r.getSigns()[axis];
				
				currentNode = left + nextNodeOffset;
				stack.push( left + (nextNodeOffset ^ 1) );
				
				continue;
			}
			else if( currentNode->getPrimitive()->rayIntersect(r,h) )
			{
					h.pos = r.parameter(h.distance);
					h.primitivePtr = currentNode->getPrimitive();
					atLeastOneHit = true;
			}
		}
		
		currentNode = stack.pop();
		if(currentNode == NULL)
			break;
	}
	

	if(showTraversalStats)
	{
		timer.stop();
		h.r = nodeTest*0;
		h.g = timer.getMicroseconds()*50;
		h.b = 10*primTest*0;
		h.skipShading = true;
	}
	
	return atLeastOneHit;
}

bool StackBVH::rayIntersectAll(const Ray &r, HitPoint &h)
{
	bool hit = intersectNodesStack(&nodes[0], r, h);
	return hit;
}

bool StackBVH::rayIntersectSingle(const Ray &r, HitPoint &h)
{
	bool hitSomething;
	//	intersects_calced++;
	
	hitSomething = h.primitivePtr->rayIntersect(r, h);
	if(hitSomething > EPSILON && hitSomething < maxFloat)
	{
		//h.distance = hitSomething;
		return true;
	}
	
	return false;
}

StackBVHNode* StackBVH::getNextNodeFromList()
{
	nodePos++;
	StackBVHNode *lastNode = &nodes[primitives.size() * 2-1];
	assert(&this->nodes[nodePos] < lastNode);
	return &this->nodes[nodePos];
}
