#include <math.h>
#include <stdio.h>
#include <time.h>
#include "../Primitives/PrimitiveBase.h"
#include "../common/Vector3.h"
#include "../common/misc.h"
#include "../common/debug.h"
#include "AABB.h"


#define numDimen 3

// intersection algorigthm from Andrew Woo
// Graphic Gems pg. 395-396
// http://tog.acm.org/GraphicsGems/gems/RayBox.c
bool AABB::rayIntersect(const Ray &r, HitPoint &h)
{
	bool inside = true;
	char quadrant[numDimen];
	slimFloat candidatePlane[numDimen];
	enum{left, middle, right};
	vector3 maxT;
	const vector3 &origin = r.getOrigin();
	const vector3 &projection = r.getDir();
	vector3 hit;
	float distance = -1;
	
	
	//find where origin lies relative to box
	for(int i=0; i<numDimen; i++)
	{
		if(origin.c[i] < bbMin.c[i])
		{
			quadrant[i] = left;
			candidatePlane[i] = bbMin[i];
			inside = false;
		}
		else if(origin.c[i] > bbMax.c[i])
		{
			quadrant[i] = right;
			candidatePlane[i] = bbMax[i];
			inside = false;
		}
		else
		{ quadrant[i] = middle; }
	}
	
	//inside the box
	if(inside)
	{
		return 1;
	}
	
	//get distances to the planes
	for(int i=0; i<numDimen; i++)
	{
		if(quadrant[i] != middle && projection.c[i] != 0.0)
			maxT[i] = (candidatePlane[i] - origin[i]) / projection.c[i];
		else
			maxT[i] = -1;
	}
	
	//get the closest plane
	int closestPlane = 0;
	for(int i=1; i<numDimen; i++)
		if(maxT[closestPlane] < maxT[i])
			closestPlane = i;
	
	if(maxT[closestPlane] < 0)
		return 0.0;
	
	for(int i=0; i<numDimen; i++)
	{
		if(closestPlane != i)
		{
			hit[i] = origin[i] + maxT[closestPlane] * projection[i];
			if(hit[i] < bbMin[i] || hit[i] > bbMax[i])
				return 0.0;
		}
		else
		{
			hit[i] = candidatePlane[i];
		}
	}
	
	distance = (hit - origin).length();
	if(distance > 0 && distance < h.distance)
	{
		//h.distance = distance;
		return true;
	}
	
	return false;
}


vector3 AABB::normalAtPoint(const vector3 &p)
{
	if(p.c[0] > bbMax.c[0] - EPSILON)
		return vector3(1,0,0);
	if(p.c[1] > bbMax.c[1] - EPSILON)
		return vector3(0,1,0);
	if(p.c[2] > bbMax.c[2] - EPSILON)
		return vector3(0,0,1);
	
	if(p.c[0] < bbMin.c[0] + EPSILON)
		return vector3(-1,0,0);
	if(p.c[1] < bbMin.c[1] + EPSILON)
		return vector3(0,-1,0);
	if(p.c[2] < bbMin.c[2] + EPSILON)
		return vector3(0,0,-1);
	
	vector3 n;
	n = p - (bbMax + bbMin)/2;
	n.normalize();
	return n;
}

void AABB::fromPoints(vector3& min, vector3& max)
{
	this->bbMin = vector3(min);
	this->bbMax = vector3(max);
}

void AABB::encompass(const AABB& box)
{
	encompass(box.bbMin, box.bbMax);
}

void AABB::encompass(const vector3& min, const vector3& max)
{
	for(int i=0; i<3; i++)
	{
		if(min[i] < bbMin[i])
			bbMin[i] = min[i];
		if(max[i] > bbMax[i])
			bbMax[i] = max[i];
	}
}

vector3 AABB::center()
{
	return (bbMin+bbMax)/2.0f;
}
