#include "debug.h"
#include "Camera.h"

#define TRACKBALL

extern SlimScene* main_scene;
void Camera::setupLookPoint()
{
	float width = main_scene->width;
	float height = main_scene->height;
	
	float imagePlaneWidth;
	float imagePlaneHeight;
	
	aspectRatio = width / height;
	horzFOV = 90.0f * M_PI/180.0f / 2.0f;
	vertFOV = 90.0f * M_PI/180.0f / 2.0f;
	imagePlaneDistance = 1.0f;
	imagePlaneWidth =  ( tan(horzFOV)*imagePlaneDistance );
	imagePlaneHeight = ( tan(horzFOV)*imagePlaneDistance ) / aspectRatio;
	spreadAcross = imagePlaneWidth / width * 2;
	spreadDown =   imagePlaneHeight / height * 2;
	
	/* set up needed vectors and points */
	/* normalize the points that we read in */
	look = vector3(lookAtPoint, pos).normalize();   //look vector
	up.normalize();
	horz = up.cross(look).normalize();   //horizontal vector
	//horz = -horz;
	vert = look.cross(horz).normalize(); //vertical vector
	
	spreadAcrossHorz = spreadAcross * horz;
	spreadDownVert = spreadDown * vert;
	
	topLeftCorner = look + (imagePlaneWidth*horz) + (imagePlaneHeight*vert);
	
	printd(DEBUG, "camera values\n");
    printd(DEBUG, "  pos=\t%f,\t%f,\t%f\n", pos.c[0],pos.c[1],pos.c[2]);
    printd(DEBUG, "  lookV=\t%f,\t%f,\t%f\n", look.c[0],look.c[1],look.c[2]);
	printd(DEBUG, "  lookat=\t%f,\t%f,\t%f\n", lookAtPoint.c[0],lookAtPoint.c[1],lookAtPoint.c[2]);
    printd(DEBUG, "  up=\t%f,\t%f,\t%f\n", up.c[0],up.c[1],up.c[2]);
    printd(DEBUG, "  horz=\t%f,\t%f,\t%f\n", horz.c[0],horz.c[1],horz.c[2]);
    printd(DEBUG, "  vert=\t%f,\t%f,\t%f\n", vert.c[0],vert.c[1],vert.c[2]);
}

void Camera::setupCamera()
{
	lookAtPoint = pos + look;
	setupLookPoint();
}

void Camera::handleInput(float xMovement, float yMovement)
{
#ifndef TRACKBALL
	vector3 xmotion;
	vector3 ymotion;
	float dis;
	float rotateAmount;
	
	//replace with real rotation later
	dis = main_scene->camera->lookAtPoint.distance(main_scene->camera->pos);
	rotateAmount = dis/50;
	
	xmotion = main_scene->camera->horz;
	ymotion = main_scene->camera->vert;
	xmotion = xmotion * xMovement * rotateAmount;
	ymotion = ymotion * yMovement * rotateAmount;
	
	main_scene->camera->lookAtPoint = main_scene->camera->lookAtPoint + xmotion;
	main_scene->camera->lookAtPoint = main_scene->camera->lookAtPoint + ymotion;
	
	setupLookPoint();
	
	main_scene->camera->pos = main_scene->camera->look;
	main_scene->camera->pos.normalize();
	main_scene->camera->pos = -main_scene->camera->pos;
	main_scene->camera->pos = main_scene->camera->pos * dis;
	
	main_scene->camera->pos = main_scene->camera->pos + main_scene->camera->lookAtPoint;
#else
	vector3 xmotion;
	vector3 ymotion;
	float dis;
	float rotateAmount;
	
	//replace with real rotation later
	dis = main_scene->camera->lookAtPoint.distance(main_scene->camera->pos);
	rotateAmount = dis/50;
	
	xmotion = main_scene->camera->horz;
	ymotion = main_scene->camera->vert;
	xmotion = xmotion * xMovement * rotateAmount;
	ymotion = ymotion * yMovement * rotateAmount;
	
	main_scene->camera->pos = main_scene->camera->pos + xmotion;
	main_scene->camera->pos = main_scene->camera->pos + ymotion;
	
	main_scene->camera->setupLookPoint();
	
	main_scene->camera->pos = main_scene->camera->look;
	main_scene->camera->pos.normalize();
	main_scene->camera->pos = -main_scene->camera->pos;
	main_scene->camera->pos = main_scene->camera->pos * dis;
	
	main_scene->camera->pos = main_scene->camera->pos + main_scene->camera->lookAtPoint;
#endif
}
void Camera::getImageCoords(vector3 &imageCoord, const vector3 &proj)
{
	float lookAmount = proj.dot(look);
	vector3 imagePlanePoint = vector3(proj) / lookAmount;
	
	imagePlanePoint = topLeftCorner - imagePlanePoint;
	float horzAmount = horz.dot(imagePlanePoint);
	float vertAmount = vert.dot(imagePlanePoint);
	
	imageCoord.c[0] = horzAmount / spreadAcross;
	imageCoord.c[1] = vertAmount / spreadDown;
}


GraphicMatrix Camera::getFrustum()
{
	//glFrustum method
	GraphicMatrix proj = GraphicMatrix::identity();
	slimFloat left = -1;
	slimFloat right = 1;
	slimFloat top = 1;
	slimFloat bottom = -1;
	slimFloat near = 0.1f;
	slimFloat far = 1000.0f;
	slimFloat a = (right + left) / (right - left);
	slimFloat b = (top + bottom) / (top - bottom);
	slimFloat c = (far + near) / (far - near);
	slimFloat d = (2 * far * near) / (far - near);
	slimFloat w = right - left;
	slimFloat h = top - bottom;
	
	proj(0,0) = (2 * near) / w;
	proj(0,2) = a;
	proj(1,1) = (2 * near) / h;
	proj(1,2) = b;
	proj(2,2) = c;
	proj(2,3) = d;
	proj(3,2) = -1.0f;
	proj(3,3) = 0.0f;
	
	return proj;
}

GraphicMatrix Camera::getProjection()
{
	GraphicMatrix proj = GraphicMatrix::identity();
	slimFloat fd = 1 / tan(vertFOV/2);
	slimFloat near = 0.1f;
	slimFloat far = 1000.0f;
	
	// DirectX method
	slimFloat w = 1;
	slimFloat h = 1;
	proj(0,0) = (2 * near) / w;
	proj(1,1) = (2 * near) / h;
	proj(2,2) = (far) / (far - near);
	proj(2,3) = (2 * far * near) / (far - near);
	proj(3,2) = -1.0f;
	proj(3,3) = 0.0f;
	
	return proj;
}

GraphicMatrix Camera::getProjectionFOV()
{
	GraphicMatrix proj = GraphicMatrix::identity();
	slimFloat fovY = M_PI/2.0f;
	slimFloat fovX = M_PI/2.0f;
	slimFloat aspect = fovX / fovY;
	slimFloat cotf2 = 1 / tan(fovY/2);
	slimFloat near = 0.1f;
	slimFloat far = 1000.0f;
	
	proj(0,0) = cotf2/aspectRatio;
	proj(1,1) = cotf2;
	proj(2,2) = (far + near) / (far - near);
	proj(2,3) = (2 * far * near) / (far - near);
	proj(3,2) = -1.0f;
	proj(3,3) = 0.0f;
	
	
	return proj;
}

GraphicMatrix Camera::getLookAt()
{
	//glLookAt method
	GraphicMatrix lookat = GraphicMatrix::identity();
	vector3 eye = this->pos;
	vector3 center = this->lookAtPoint;
	vector3 up = this->up;
	vector3 F = center - eye;
	F.normalize();
	vector3 s = F.cross(up);
	s.normalize();
	vector3 u = s.cross(F);
	
	//IBM's wacko spec
	lookat.setCol(0, s);
	lookat.setCol(1, u);
	lookat.setCol(2, -F);
	//lookat.setCol(3, -eye);
	lookat(3,3) = 1.0f;

	//from openGL spec
	lookat.setRow(0, s);
	lookat.setRow(1, u);
	lookat.setRow(2, -F);
	//lookat.setCol(3, -eye);
	lookat(3,3) = 1.0f;
	
	vector3 inveye = -eye;
	GraphicMatrix trans = GraphicMatrix::translation(inveye);
	return lookat*trans;
}

GraphicMatrix Camera::getPerspectiveScaling()
{
	//glLookAt method
	GraphicMatrix lookat = GraphicMatrix::identity();
	vector3 e = lookAtPoint;
	
	lookat(0,3) = -e[0];
	lookat(1,3) = -e[1];
	lookat(3,2) = 1.0f/e[2];
	lookat(3,3) = 0.0f;
	
	return lookat;
}
