#ifndef __SIMPLEBOUNDINGBOX
#define __SIMPLEBOUNDINGBOX

#include "../common/Common.h"
#include "../common/Vector3.h"
#include "../common/Ray.h"
#include "../common/HitPoint.h"

class SimpleBoundingBox
{
public:
	vector3 bbMin, bbMax;
	
	SimpleBoundingBox()
	{
		this->bbMin = vector3(maxFloat,maxFloat,maxFloat);
		this->bbMax = vector3(minFloat,minFloat,minFloat);
	}
	
	SimpleBoundingBox(vector3 centerPoint, slimFloat extent)
	{
		this->bbMin = centerPoint - extent;
		this->bbMax = centerPoint + extent;
	}
	
	SimpleBoundingBox(vector3 min, vector3 max)
	{
		this->bbMin = min;
		this->bbMax = max;
	}
	
	~SimpleBoundingBox()
	{
		//free(children);
	}
	
	
	inline slimFloat minf(const slimFloat a, const slimFloat b)
	{ return a < b ? a : b; }
	inline slimFloat maxf(const slimFloat a, const slimFloat b)
	{ return a > b ? a : b; }
	
	// intersection method based on mueller geimer by tbp from http://ompf.org/ray/ray_box.html
	bool tbpIntersect(const Ray &r, HitPoint &h)
	{
		slimFloat lambda1, lambda2, lmin, lmax;
		lmin = minFloat;
		lmax = maxFloat;
		
		lambda1 = (bbMin[0] - r.getOrigin()[0]) * r.getReciprocal()[0];
		lambda2 = (bbMax[0] - r.getOrigin()[0]) * r.getReciprocal()[0];
		lmin = minf(lambda1, lambda2);
		lmax = maxf(lambda1, lambda2);
		if(lmin > lmax) return false;
		
		lambda1 = (bbMin[1] - r.getOrigin()[1]) * r.getReciprocal()[1];
		lambda2 = (bbMax[1] - r.getOrigin()[1]) * r.getReciprocal()[1];
		lmin = maxf(minf(lambda1, lambda2), lmin);
		lmax = minf(maxf(lambda1, lambda2), lmax);
		if(lmin > lmax)	return false;
		
		lambda1 = (bbMin[2] - r.getOrigin()[2]) * r.getReciprocal()[2];
		lambda2 = (bbMax[2] - r.getOrigin()[2]) * r.getReciprocal()[2];
		lmin = maxf(minf(lambda1, lambda2), lmin);
		lmax = minf(maxf(lambda1, lambda2), lmax);
		
		return (lmin <= lmax);
		//return ((lmax > 0.f) & (lmax > lmin));
		//return ((lmax >= 0.f) & (lmax >= lmin));
	}

	// intersection method based on williams "an efficient and robust ray-box intersection algorithm"
	bool williamsIntersect(const Ray &r, HitPoint &h)
	{
		slimFloat tmin, tmax, txmin, txmax, tymin, tymax, tzmin, tzmax;
		vector3 bounds[2];
		bounds[0] = bbMin;
		bounds[1] = bbMax;
		
		tmin = minFloat;
		tmax = maxFloat;
		
		txmin = (bounds[r.getSigns()[0]].c[0] - r.getOrigin()[0]) * r.getReciprocal()[0];
		txmax = (bounds[r.getSigns()[3]].c[0] - r.getOrigin()[0]) * r.getReciprocal()[0];
		if( txmin > txmax)
			return false;
		tmin = txmin;
		tmax = txmax;
		
		tymin = (bounds[r.getSigns()[1]].c[1] - r.getOrigin()[1]) * r.getReciprocal()[1];
		tymax = (bounds[r.getSigns()[4]].c[1] - r.getOrigin()[1]) * r.getReciprocal()[1];
		if( (tmin > tymax) || (tymin > tmax) )
			return false;
		tmin = maxf(tymin, tmin);
		tmax = minf(tymax, tmax);
		
		tzmin = (bounds[r.getSigns()[2]].c[2] - r.getOrigin()[2]) * r.getReciprocal()[2];
		tzmax = (bounds[r.getSigns()[5]].c[2] - r.getOrigin()[2]) * r.getReciprocal()[2];
		if( (tmin > tzmax) || (tzmin > tmax) )
			return false;
		tmin = maxf(tzmin, tmin);
		tmax = minf(tzmax, tmax);
		
		return ( (tmin < h.distance) && (tmax >= 0.0f) );
	}
	
	// better intersect - combines williams and fyffe
	bool signedIntersect(const Ray &r, HitPoint &h)
	{
		slimFloat tmin, tmax, txmin, txmax, tymin, tymax, tzmin, tzmax;
		const float *b = &bbMin[0];
		
		txmin = (b[r.lowIndexX]  - r.getOrigin()[0]) * r.getReciprocal()[0];
		txmax = (b[r.highIndexX] - r.getOrigin()[0]) * r.getReciprocal()[0];
		if( txmin > txmax)
			return false;
		tmin = txmin;
		tmax = txmax;
		
		tymin = (b[r.lowIndexY]  - r.getOrigin()[1]) * r.getReciprocal()[1];
		tymax = (b[r.highIndexY] - r.getOrigin()[1]) * r.getReciprocal()[1];
		if( (tmin > tymax) || (tymin > tmax) )
			return false;
		tmin = maxf(tymin, tmin);
		tmax = minf(tymax, tmax);
		
		tzmin = (b[r.lowIndexZ]  - r.getOrigin()[2]) * r.getReciprocal()[2];
		tzmax = (b[r.highIndexZ] - r.getOrigin()[2]) * r.getReciprocal()[2];
		if( (tmin > tzmax) || (tzmin > tmax) )
			return false;
		tmin = maxf(tzmin, tmin);
		tmax = minf(tzmax, tmax);
		
		return ( (tmin < h.distance) && (tmax >= 0.0f) );
	}
	
	// from Graham Fyffe, Ray tracing news, November 8, 2008, Volume 21, Number 1
	// http://tog.acm.org/resources/RTNews/html/rtnv21n1.html#art9
	bool fyffeIntersection(const Ray &r, HitPoint &h)
	{
		const float *b = &bbMin[0];
		float t1x = (b[r.lowIndexX] - r.getOrigin()[0]) * r.getReciprocal()[0];
		float t2x = (b[r.highIndexX] - r.getOrigin()[0]) * r.getReciprocal()[0];
		float t1y = (b[r.lowIndexY] - r.getOrigin()[1]) * r.getReciprocal()[1];
		float t2y = (b[r.highIndexY] - r.getOrigin()[1]) * r.getReciprocal()[1];
		float t1z = (b[r.lowIndexZ] - r.getOrigin()[2]) * r.getReciprocal()[2];
		float t2z = (b[r.highIndexZ] - r.getOrigin()[2]) * r.getReciprocal()[2];
		if (t1x > t2y || t2x < t1y || t1x > t2z || t2x < t1z || t1y > t2z || t2y < t1z) return false;
		if (t2x < 0.0 || t2y < 0.0 || t2z < 0.0) return false;
		// [NOTE] uncomment next line if t stores the ray's current length
		if (t1x > h.distance || t1y > h.distance || t1z > h.distance) return false;
		
		// the next four lines are optional, they compute the intersection distance and return it
		//float distance = t1x;
		//if (t1y > distance) distance = t1y;
		//if (t1z > distance) distance = t1z;
		//*t = distance;
		return true;
	}
	
	void encompass(const SimpleBoundingBox& box)
	{
		encompass(box.bbMin, box.bbMax);
	}
	
	void encompass(const vector3& min, const vector3& max)
	{
		for(int i=0; i<3; i++)
		{
			if(min[i] < bbMin[i])
				bbMin[i] = min[i];
			if(max[i] > bbMax[i])
				bbMax[i] = max[i];
		}
	}
	
	vector3 center()
	{ return (bbMin+bbMax)*0.5f; }
	
	vector3 dimension()
	{ return (bbMax-bbMin); }

	slimFloat surfaceArea()
	{
		vector3 dim = this->dimension();
		slimFloat area = dim[0]*dim[1] + dim[1]*dim[2] + dim[2]*dim[0];
		return area * 2;
	}
};

#endif
