#ifndef __VECTOR3
#define __VECTOR3
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

// inlining can be adjusted here
// #define INLINE_MODE inline
#define INLINE_MODE

// floating point unit can be changed here
#define v3float float
// #define v3float float

// set an epsilon if not set
#ifndef EPSILON
#define EPSILON 0.000000001f
#endif

template <int dimension>
class SlimVector
{
public:

	static const int dim = dimension;
	v3float c[dimension]; //compenents x, y, z
	
	// SlimVector creation
	SlimVector()
	{
		for(int i=0; i<dim; i++)
			c[i] = 0.0f;
	}
	
	SlimVector(const SlimVector source[])
	{
		for(int i=0; i<dim; i++)
			c[i] = source.c[i];
	}
	
	SlimVector(const v3float *source)
	{
		for(int i=0; i<dim; i++)
			c[i] = source[i];
	}
	
	SlimVector(const v3float x, const v3float y, const v3float z)
	{ c[0] = x; c[1] = y; c[2] = z; }
	
	SlimVector(const SlimVector &to, const SlimVector &from)
	{
		for(int i=0; i<dim; i++)
			c[i] = to.c[i] - from.c[i];
	}
	
	// operator overload
	INLINE_MODE SlimVector operator+(const SlimVector &v1) const
	{
		SlimVector t;
		for(int i=0; i<dim; i++)
			t.c[i] = c[i] + v1.c[i];
		return t;
	}
	
	INLINE_MODE SlimVector operator-(const SlimVector &v1) const
	{
		SlimVector t;
		for(int i=0; i<dim; i++)
			t.c[i] = c[i] - v1.c[i];
		return t;
	}
	
	INLINE_MODE SlimVector operator+(const v3float f) const
	{
		SlimVector t;
		for(int i=0; i<dim; i++)
			t.c[i] = c[i] + f;
		return t;
	}
	
	INLINE_MODE SlimVector operator-(const v3float f) const
	{
		SlimVector t;
		for(int i=0; i<dim; i++)
			t.c[i] = c[i] - f;
		return t;
	}
	
	INLINE_MODE SlimVector operator*(const v3float f) const
	{
		SlimVector t;
		for(int i=0; i<dim; i++)
			t.c[i] = c[i] * f;
		return t;
	}
	
	INLINE_MODE SlimVector operator/(const v3float f) const
	{
		SlimVector t;
		for(int i=0; i<dim; i++)
			t.c[i] = c[i] / f;
		return t;
	}
	
	INLINE_MODE SlimVector& operator+=(const SlimVector &v1)
	{
		for(int i=0; i<dim; i++)
			c[i] += v1.c[i];
		return *this;
	}
	
	INLINE_MODE SlimVector& operator-=(const SlimVector &v1)
	{
		for(int i=0; i<dim; i++)
			c[i] -= v1.c[i];
		return *this;
	}
	
	INLINE_MODE SlimVector& operator+=(const v3float f)
	{
		for(int i=0; i<dim; i++)
			c[i] += f;
		return *this;
	}
	
	INLINE_MODE SlimVector& operator-=(const v3float f)
	{
		for(int i=0; i<dim; i++)
			c[i] -= f;
		return *this;
	}
	
	INLINE_MODE SlimVector& operator*=(const v3float f)
	{
		for(int i=0; i<dim; i++)
			c[i] *= f;
		return *this;
	}
	
	INLINE_MODE SlimVector& operator/=(const v3float f)
	{
		for(int i=0; i<dim; i++)
			c[i] /= f;
		return *this;
	}
	
	INLINE_MODE SlimVector& operator=(const SlimVector &v1)
	{
		for(int i=0; i<dim; i++)
			c[i] = v1.c[i];
		return *this;
	}
	
	INLINE_MODE bool operator==(const SlimVector &v1)
	{
		bool equal = true;
		for(int i=0; i<dim; i++)
			equal = equal && c[i] == v1.c[i];
		return equal;
	}
	
	INLINE_MODE SlimVector operator-()
	{
		SlimVector t;
		for(int i=0; i<dim; i++)
			t.c[i] = -c[i];
		return t;
	}
	
	INLINE_MODE v3float operator[](const int i) const
	{
		return c[i];
	}
	
	INLINE_MODE v3float& operator[](const int i)
	{
		return c[i];
	}
	
	// arithmetic operations
	INLINE_MODE v3float dot(const SlimVector &a) const
	{
		return a.c[0]*c[0] + a.c[1]*c[1] + a.c[2]*c[2];
	}

	INLINE_MODE SlimVector toInt()
	{
		SlimVector t;
		for(int i=0; i<dim; i++)
			t.c[i] = (int)c[i];
		return t;
	}
	
	
	INLINE_MODE SlimVector cross(const SlimVector &a) const
	{
		SlimVector dest;
		dest.c[0] = c[1]*a.c[2] - c[2]*a.c[1];
		dest.c[1] = c[2]*a.c[0] - c[0]*a.c[2];
		dest.c[2] = c[0]*a.c[1] - c[1]*a.c[0];
		return dest;
	}
	
	// SlimVector unit operations
	INLINE_MODE v3float length() const
	{
		v3float sqauredLength = 0.0f;
		for(int i=0; i<dim; i++)
			sqauredLength += c[i] * c[i];
		return sqrt(sqauredLength);
	}
	
	INLINE_MODE SlimVector& normalize()
	{
		v3float normalizeLength;
		normalizeLength = this->length();
		
		if(normalizeLength <= EPSILON)
		{
			//printf("cannot normalize degenerate SlimVector\n");
			return *this;
		}
		
		*this /= normalizeLength;
		return *this;
	}
	
	INLINE_MODE void print() const
	{
		for(int i=0; i<dim; i++)
			printf("%.2f ", c[i]);
		printf("\n");
	}
	
	// SlimVector creation
	INLINE_MODE SlimVector& randomize()
	{
		do {
			c[0] = (v3float)rand()/RAND_MAX*2-1;
			c[1] = (v3float)rand()/RAND_MAX*2-1;
			c[2] = (v3float)rand()/RAND_MAX*2-1;
		} while (c[0]*c[0] + c[1]*c[1] + c[2]*c[2] > 1.0);
		return *this;
	}
	
	INLINE_MODE SlimVector& randomizeFast()
	{
		for(int i=0; i<dim; i++)
			c[i] = (v3float)rand()/RAND_MAX;
		return *this;
	}
	
	// SlimVector combination operations
	INLINE_MODE v3float distanceSquared(const SlimVector &a)
	{
		v3float d = 0.0;
		for(int i=0; i<dim; i++)
			d += pow(a.c[i] - c[i], 2);
		return d;
	}

	INLINE_MODE v3float distance(const SlimVector& a)
	{
		v3float disSquared = this->distanceSquared(a);
		return sqrt( disSquared );
	}
	
	INLINE_MODE v3float angle(const SlimVector &a)
	{
		return acos(this->dot(a) / this->length() / a.length());
	}
	
	INLINE_MODE SlimVector reflect(const SlimVector &normal)
	{
		v3float dp;
		SlimVector dest;
		
		dp = 2*this->dot(normal);
		dest = *this - (normal*dp);
		return dest;
	}
	
	INLINE_MODE SlimVector reflectRandom(SlimVector& norm, v3float spread)
	{
		SlimVector random1;
		SlimVector perturb;
		v3float perturb_amount = (float)rand()/RAND_MAX * (1.0 - spread);
		SlimVector reflection = this->reflect(norm);
		
		random1.randomize();
		perturb = reflection.cross(random1);
		perturb.normalize();
		
		//scale vectors for the appropriate spread
		perturb = perturb * perturb_amount;
		reflection = reflection * (1.0f - perturb_amount);
		
		//perturb the perfect reflection
		reflection = reflection + perturb;
		reflection.normalize();
		
		return reflection;
	}
	
	INLINE_MODE SlimVector diffuseScatter(const SlimVector &normal)
	{
		SlimVector dest;
		
		do {
			dest.randomize();
		} while (dest.dot(normal) < 0);
		return dest;
	}
	
	INLINE_MODE v3float getIndexEntering(SlimVector& norm, v3float hitIndex)
	{
		char inside = this->dot(norm) > 0;
		if(inside)
			return 1.0f;
		return hitIndex;
	}
	
	INLINE_MODE v3float getIndexExiting(SlimVector& norm, v3float hitIndex)
	{
		char inside = this->dot(norm) > 0;
		if(inside)
			return hitIndex;
		return 1.0f;
	}
	
	INLINE_MODE v3float refract(SlimVector& refractVectorOut, SlimVector& norm, v3float hitIndex)
	{
		SlimVector i = *this;
		SlimVector n = norm;
		SlimVector i_inv = SlimVector(-i);
		
		i.normalize();
		n.normalize();
		i_inv.normalize();
		
		//if(debug_ray)
		//	printf("debugger");
		
		//get the index of the material we're exiting
		const v3float n1 = this->getIndexExiting(n, hitIndex);
		const v3float n2 = this->getIndexEntering(n, hitIndex);
		const v3float ndiv = n1/n2;
		const v3float inside = n.dot(i_inv);
		const v3float criticalRatio = n2/n1;
		const v3float cosTheta1 = (inside>0) ? inside : -inside;
		const v3float cosTheta2 = sqrt( 1 - pow(n1/n2,2) * (1 - pow(cosTheta1, 2)));
		
		//reflection stuff
		const v3float theta1 = acosf(cosTheta1);
		const v3float criticalAngle = (criticalRatio < 1) ? asinf(criticalRatio) : 0.0001;
		v3float reflectAmount = 1.0f;
		if(criticalRatio > 1)
			reflectAmount = 0.0f;
		if(theta1 < criticalAngle)
			reflectAmount = theta1 / criticalAngle;
		
		//refraction
		if(cosTheta1 < 0)
			refractVectorOut = i*(ndiv) + n * (ndiv * cosTheta1 + cosTheta2);
		else
			refractVectorOut = i*(ndiv) + n * (ndiv * cosTheta1 - cosTheta2);
		refractVectorOut.normalize();
		return (float)reflectAmount;
	}

	INLINE_MODE int maxComponent()
	{
		int largest = 0;
		for(int i=1; i<dim; i++)
			if(c[i] > c[i-1])
				largest = i;
		return largest;
	}
};

template <int dimension>
static INLINE_MODE SlimVector<dimension> operator*(const v3float f, const SlimVector<dimension> &v1)
{
	return v1*f;
}

template <int dimension>
static INLINE_MODE SlimVector<dimension> operator/(const v3float f, const SlimVector<dimension> &v1)
{
	SlimVector<dimension> t;
	for(int i=0; i<v1.dim; i++)
		t[i] = f / v1[i];
	return t;
}

typedef SlimVector<2> vector2;
typedef SlimVector<3> vector3;
typedef SlimVector<4> vector4;

static vector4 v4fromv3(const vector3 &v3)
{
	v3float c[4] = {1,1,1,1};
	vector4 v4(c);
	for(int i=0; i<v3.dim; i++)
		v4[i] = v3[i];
	return v4;
}

/*
class vector3 : public SlimVector<3>
{
public:
	vector3() : SlimVector<3>(){};
	vector3(const vector3 &source) : SlimVector<3>(source){};
	vector3(const v3float x, const v3float y, const v3float z) : SlimVector<3>(x,y,z){};
	vector3(const vector3 &to, const vector3 &from) : SlimVector<3>(to, from){};
//	{ c[0] = x; c[1] = y; c[2] = z; }
	
	using SlimVector<3>::operator=;
};


static INLINE_MODE vector3 operator*(const v3float f, const vector3 &v1)
{
	vector3 t = v1;
	t = t * f;
	return t;
}

static INLINE_MODE vector3 operator/(const v3float f, const vector3 &v1)
{
	v3float fdiv = 1.0f/f;
	return v1 * fdiv;
}
 */

#endif
