#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "../Primitives/PrimitiveBase.h"
#include "../common/Vector3.h"

#include "../common/scene.h"
#include "../common/scene.h"
#include "../common/debug.h"
#include "phong.h"
#include "lighting.h"

#include "../common/Color.h"

extern SlimScene* main_scene;

//int shadow_cache_id = -1;
PrimitiveBase *shadowCachePtr = NULL;
int shadow_cache_max_misses = 5;
int shadow_cache_misses = 0;
//#define USE_SHADOW_CACHE

//float shadow_cache_test(vector3 *origin,vector3 *light_dir, float distance_to_light)
inline bool lightObstructed(vector3& origin, vector3& light_dir, float distance_to_light, PrimitiveBase *testobj)
{
	vector3 new_intersection;	
	Ray r(origin, light_dir);
	HitPoint h;
	
#ifdef USE_SHADOW_CACHE
	// cached Primitive test
	if(shadowCachePtr != NULL)
	{
		h.primitivePtr = shadowCachePtr;
		if(main_scene->hierarchy->rayIntersectSingle(r, h) && h.distance > EPSILON && h.distance < distance_to_light)
		{
			shadow_cache_misses = 0;
			return true;
		}
	}
#endif
	
	// all Primitive test
	if (main_scene->hierarchy->rayIntersectAll(r, h) && h.distance < distance_to_light )
	//if (intersect_all_Primitives_getptr(r, h) && h.distance < distance_to_light )
	{
#ifdef USE_SHADOW_CACHE
		shadowCachePtr = h.primitivePtr;
		shadow_cache_misses = 0;
#endif
		return true;
	}

#ifdef USE_SHADOW_CACHE
	// not hits, so no shadow	
	if(shadow_cache_misses > shadow_cache_max_misses)
	{
		shadowCachePtr = NULL;
	}
	else
	{
		shadow_cache_misses++;
	}
#endif

	return false;
}


inline void light_disc_shadow(intersect_data *id, Color *shadowed_Color)
{
	vector3 light_dir;
	vector3 light_pos;
	vector3 random_v;
	vector3 &hit_point = id->intersect;
	int rays_tried = 0;
	int visible_rays = 0;
	float radius = main_scene->lights[id->other_obj_num]->radius;
	float last_shadow_percent = 1.0;
	float allowed_variance = SHADOW_SAMPLE_VARIANCE;
	int check_every = SHADOW_CHECK_EVERY;
	int next_check = check_every;
	float distance_to_light;
	float distance_closest_Primitive;
	
	for(rays_tried = 1; rays_tried <= SHADOW_RAY_SAMPLES; rays_tried++)
	{
		//random point on light surface
		random_v.randomize();
		light_pos = random_v.cross(main_scene->lights[id->other_obj_num]->norm);
		light_pos.normalize();  //some vector in disc plane now

		//lastly,vector3 needs a random length
		light_pos = light_pos * radius;
		//FIXME: need to multiply by random number!
		//vector_scalar_subtract(&light_pos, radius);

		light_pos = light_pos + main_scene->lights[id->other_obj_num]->pos;

		//from the Primitive to the light
		light_dir = vector3(light_pos, hit_point);
		light_dir.normalize();


		distance_to_light = hit_point.distance(light_pos);
		distance_closest_Primitive = lightObstructed(hit_point, light_dir, distance_to_light, id->obj);
		if(distance_closest_Primitive == 0.0f)
		{
			visible_rays++;
		}
		
		shadowed_Color->g = (float)rays_tried;
		
		//shadow adaptively
		//keep going only if the shadow is getting darker
		if(rays_tried == next_check)
		{
			shadowed_Color->r = (float)visible_rays / (float)rays_tried;
			if(last_shadow_percent == shadowed_Color->r)
			{
				return;
			}
			float diff = fabs(last_shadow_percent - shadowed_Color->r);
			if(diff < allowed_variance)
			{
				return;
			}
			last_shadow_percent = 	shadowed_Color->r;
			next_check += check_every;
		}
	}

	shadowed_Color->r = (float)visible_rays / (float)(rays_tried-1);
}


inline void light_sphere_shadow(intersect_data *id, Color *shadowed_Color)
{
	vector3 light_dir;
	vector3 light_pos;
	//vector3 shadow_hit;
	vector3& hit_point = id->intersect;
	int rays_tried = 0;
	int visible_rays = 0;
	float two_radius = main_scene->lights[id->other_obj_num]->radius * 2;
	float radius = main_scene->lights[id->other_obj_num]->radius;
	float last_shadow_percent = 1.0;
	float allowed_variance = 0.0;
	int check_every = 100;
	int next_check = check_every;
	float distance_to_light;
	float distance_closest_Primitive;
	
	for(rays_tried = 1; rays_tried <= SHADOW_RAY_SAMPLES; rays_tried++)
	{
		//random point on light surface
		light_pos.randomizeFast();
		light_pos = light_pos * two_radius;
		light_pos = light_pos - radius;
		light_pos = light_pos + main_scene->lights[id->other_obj_num]->pos;

		//from the Primitive to the light
		light_dir = vector3(light_pos, hit_point);
		light_dir.normalize();


		distance_to_light = hit_point.distance(light_pos);
		distance_closest_Primitive = lightObstructed(hit_point, light_dir, distance_to_light, id->obj);
		if(distance_closest_Primitive == 0.0f)
		{
			visible_rays++;
		}
		
		shadowed_Color->g = (float)rays_tried;
		
		//shadow adaptively
		//keep going only if the shadow is getting darker
		if(rays_tried == next_check)
		{
			shadowed_Color->r = (float)visible_rays / (float)rays_tried;
			if(last_shadow_percent == shadowed_Color->r)
			{
				return;
			}

			if(fabs(last_shadow_percent - shadowed_Color->r) < allowed_variance)
			{
				return;
			}
			last_shadow_percent = 	shadowed_Color->r;
			next_check += check_every;
		}
	}

	shadowed_Color->r = (float)visible_rays / (float)(rays_tried-1);
}

inline void light_point_shadow(intersect_data *id, Color *shadowed_Color)
{
	vector3& origin = id->intersect;
	vector3 light_dir;
	float distance_to_light;
	bool obstructed;
	
	light_dir = main_scene->lights[id->other_obj_num]->pos - origin;
	light_dir.normalize();
	
	distance_to_light = origin.distance(main_scene->lights[id->other_obj_num]->pos);
	obstructed = lightObstructed(origin, light_dir, distance_to_light, id->obj);
	if(obstructed)
	{
		shadowed_Color->r = 0.0f;
		return;
	}

	shadowed_Color->r = 1.0f;
	return;
}



inline void light_infinite_shadow(intersect_data *id, Color *shadowed_Color)
{
//	PrimitiveBase *ob;
//	vector3 tracePt;	//1st Primitive on ray from Primitive to light
//	vector3 *intersectP = &id->intersect;

	//light is at infinity, so the pos is the light vector
	//the position should already be normalized

	// see if there is an Primitive from intersection to light
	//ob = intersect(intersectP, &main_scene->lights[light_num]->pos, &tracePt, 1);

	//if there is any Primitive, it block the light
	//if ( ob != NULL ) 
	//	return 0;
	shadowed_Color->r = 1;
	return;
}


inline void light_spotlight_shadow(intersect_data *id, Color *shadowed_Color)
{
	vector3 lightV;
	vector3 lightP;
	PrimitiveBase *ob;
	vector3 spotV;
	vector3 tracePt;	//1st Primitive on ray from Primitive to light
	vector3 &intersectP = id->intersect;
	//float dis = 0.5;
	//int i;
	float LdpL;


	/* light vector */
	lightV.c[0] = main_scene->lights[id->other_obj_num]->pos.c[0] - intersectP.c[0];
	lightV.c[1] = main_scene->lights[id->other_obj_num]->pos.c[1] - intersectP.c[1];
	lightV.c[2] = main_scene->lights[id->other_obj_num]->pos.c[2] - intersectP.c[2];
	lightV.normalize();
	
	spotV = vector3(main_scene->lights[id->other_obj_num]->pos, main_scene->lights[id->other_obj_num]->norm);
	spotV.normalize();
	LdpL = lightV.dot( spotV);

	if( LdpL <= 0 || LdpL < 0 )
	{
		shadowed_Color->r = 0;
		return;
	}
	
	//scale
	LdpL = 1 - spotV.distance(lightV) / main_scene->lights[id->other_obj_num]->radius;
	if(LdpL > 1)
	{
		shadowed_Color->r = 0;
		return;
	}

	LdpL = LdpL * main_scene->lights[id->other_obj_num]->radius;

	lightP.c[0] = main_scene->lights[id->other_obj_num]->pos.c[0];
	lightP.c[1] = main_scene->lights[id->other_obj_num]->pos.c[1];
	lightP.c[2] = main_scene->lights[id->other_obj_num]->pos.c[2];


	/* calculate shadow */
	//ob = intersect_all_Primitives_getptr(intersectP, lightV, tracePt);

	/* if we are in a shadow, try the next light (ambient is the only
	 * lighting needed and it is already done */
	if ( ob != NULL && intersectP.distance(tracePt) <
			intersectP.distance(lightP))
		shadowed_Color->r = 0;
	else
		shadowed_Color->r = LdpL * 1;

	return;
}


void get_shadow(intersect_data *id, Color *shadowed_Color)
{
	shadowed_Color->r = 1.0f;

	if (main_scene->lights[id->other_obj_num]->obj_type == LIGHT_INFINITE)
		//shadow = pointlight(light_num, id);
		//shadow = 1;
		return;

	else if( main_scene->lights[id->other_obj_num]->obj_type == LIGHT_SPOT)
		light_spotlight_shadow(id, shadowed_Color);

	else if( main_scene->lights[id->other_obj_num]->obj_type == LIGHT_POINT)
		light_point_shadow(id, shadowed_Color);

	else if (main_scene->lights[id->other_obj_num]->obj_type == LIGHT_SPHERE)
		light_sphere_shadow(id, shadowed_Color);
	
	else if (main_scene->lights[id->other_obj_num]->obj_type == LIGHT_DISC)
		light_disc_shadow(id, shadowed_Color);
#ifdef DEBUG_CACHE
shadowed_Color->r = 1.0f;
#endif
//	return 1.0f;
	//return shadow;
}
